//*************************************************************************************************
//
//	Description:
//		garage_floor.fx - A glossy reflective garage floor.
//
//	<P> Copyright (c) 2009 Slightly Mad Studios Ltd. All rights reserved.
//
//	Author: 
//		Tom Nettleship
//
//	History:
//
//	<TABLE>
//		\Author         Date        Version       Description
//		--------        -----       --------      ------------
//		TNettleship     10/04/2009  0.1           Created
//	<TABLE>
//
//*************************************************************************************************
#define _SSAO_READY_

#include "stddefs.fxh"
#include "specialisation_globals.fxh"


//-----------------------------------------------------------------------
//
// Preprocessor definitions
//

#define MAX_ENVMAP_BIAS	6.0f
#define AMBIENT_SHADOW_STRENGTH 0.667f


// Compiler test settings, exercises all options
#if defined( TEST_COMPILE )
#define NORMAL_MAPPING
#endif



//-----------------------------------------------------------------------
//
// Input parameters
//

//
// Camera
//
#ifdef _3DSMAX_
// 3DSMax parser 0x0001 doesn't support WorldCameraPosition, so we need to bring the view matrix
// in to access the 4th row to get the same information. Parser 0x0000 supports it. Bleh.
float4x4 viewI : ViewInverse
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
// The ingame renderer directly supplies the camera position
SHARE_PARAM float3 worldCameraPos : WorldCameraPosition
<
	string UIWidget = "None";
	bool appEdit = false;
>;
#endif



//
// Transforms
//
#if defined( _3DSMAX_ ) || defined(_PS3_)
// Max doesn't support viewproj as an app-supplied parameter
SHARE_PARAM float4x4 worldviewproj : WorldViewProjection
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
>;
#else
SHARE_PARAM float4x4 viewproj : ViewProjection
<
	bool appEdit = false;
	bool export = false;
>;
#endif

float4x4 world : World
<
	string UIWidget = "None";
	bool appEdit = false;
	bool export = false;
	bool dynamic = true;
>;



//
// Channel mappings (max only)
//

//
// N.B. Max contains a bug which means the colour channel must NOT be mapped to texcoord0.
// The first UV coord channel MUST be mapped to texcoord0 or the basis vectors for normal
// mapping will be screwed up. (e.g. there's some bit of code deep within max which assumes
// this setup when calculating the basis vectors)
//

#ifdef _3DSMAX_

// First UV channel
int texcoord0 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 0;
	int MapChannel = 1;
	int RuntimeTexcoord = 0;
	bool export = false;
> = 0;

// Vertex colour channel
int texcoord1 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 1;
	int MapChannel = 0;
	bool ColorChannel = true;
	bool export = false;
> = 0;

// Vertex alpha channel (max presents it seperately for no good reason)
int texcoord2 : Texcoord
<
	string UIWidget = "None";
	int Texcoord = 2;
	int MapChannel = -2;
	bool ColorChannel = true;
	bool export = false;
>;

#endif


//
// Textures
//

#ifdef _3DSMAX_
texture diffuseTexture : DiffuseMap						// Diffuse colour in RGB, translucency in alpha
#else
texture diffuseTexture : TEXTURE							// Diffuse colour in RGB, translucency in alpha
#endif
<
	string UIName = "Diffuse Tex {UV1}";
	bool appEdit = true;
	bool export = true;
>;

texture specularTexture : TEXTURE							// Specular colour in RGB, power interpolation in alpha
<
	string UIName = "Spec Tex {UV1}";
	bool appEdit = true;
	bool export = true;
>;

SPECIALISATION_PARAM( useNormalMap, "Use normal map?", "NORMAL_MAPPING" )	// TRUE if the normal map is to be used in lighting

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
DEPENDENT_TEXTURE_PARAM( normalTexture, "Normal Tex {UV1}", useNormalMap )
#endif

#if !defined( _3DSMAX_ )
texture reflectionTexture : TEXTURE
<
	string UIName = "Env Texture";
	bool appEdit = true;
>;

texture ambientShadowTexture : TEXTURE							// Ambient shadow strength bitmap
<
	string UIName = "Ambient Tex";
	bool appEdit = true;
>;
#endif

float ambientXSize
<
	string UIName = "Ambient X Size";
	bool appEdit = true;
	bool export = true;
> = 2.0f;

float ambientZSize
<
	string UIName = "Ambient Z Size";
	bool appEdit = true;
	bool export = true;
> = 4.0f;

float fresnelFactor
<
	string UIName = "Fresnel Factor";
	bool appEdit = true;
	bool export = true;
> = 0.6f;

float minSpecPower
<
	string UIName = "Min Specular Power";
	bool appEdit = true;
	bool export = true;
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
> = 1.0f;

float maxSpecPower
<
	string UIName = "Max Specular Power";
	bool appEdit = true;
	bool export = true;
	float UIMin = 1.0f;
	float UIMax = 1024.0f;
> = 32.0f;

float globalSpecularFactor
<
	string UIName = "Specular Factor";
	bool appEdit = true;
	bool export = true;
> = 1.0f;

float reflectionFactor
<
	string UIName = "Refl Factor";
	bool appEdit = true;
	bool export = true;
> = 1.0f;

//
// Lighting
//

#include "lighting_globals.fxh"
DECLARE_LIGHTING_PARAMS



//-----------------------------------------------------------------------
//
// Samplers
//

sampler2D diffuseMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="diffuseTexture"; 
	string MinFilter = "Anisotropic";
	string MagFilter = "Anisotropic";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
	int MaxAnisotropy = 16;
> 
= sampler_state
{
	Texture = < diffuseTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _ANISOMINFILTER;
	MagFilter = _ANISOMAXFILTER;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_MAX_ANISOTROPY( 16 )
#endif
};

sampler2D specularMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="specularTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < specularTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
sampler2D normalMap : SAMPLER 
< 
	SET_LINEAR_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="normalTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < normalTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
	AddressU  = Wrap;
	AddressV  = Wrap;
#if defined(_PS3_)
	LODBias = 0;
#else
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


#if !defined( _3DSMAX_ )
sampler2D reflectionMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="reflectionTexture";
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
//	string AddressU  = "Clamp";
//	string AddressV  = "Clamp";
	string AddressU  = "Wrap";
	string AddressV  = "Wrap";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < reflectionTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_SRGB_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	LODBias = 0;
#else
//	AddressU  = Clamp;
//	AddressV  = Clamp;
	AddressU  = Wrap;
	AddressV  = Wrap;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};

sampler2D ambientShadowMap : SAMPLER 
< 
	SET_SRGB_TEXTURE
	bool appEdit = false; 
	string SamplerTexture="ambientShadowTexture"; 
	string MinFilter = "Linear";
	string MagFilter = "Linear";
	string MipFilter = "Linear";
	string AddressU  = "Clamp";
	string AddressV  = "Clamp";
	int MipMapLODBias = 0;
> 
= sampler_state
{
	Texture = < ambientShadowTexture >;
#if defined(SET_FX_SAMPLER_STATES)
	FX_SAMPLERSTATE_LINEAR_TEXTURE
	MinFilter = _MINFILTER;
	MagFilter = Linear;
	MipFilter = Linear;
#if defined(_PS3_)
	AddressU  = ClampToEdge;
	AddressV  = ClampToEdge;
	LODBias = 0;
#else
	AddressU  = Clamp;
	AddressV  = Clamp;
	MipMapLODBias = 0;
#endif
	SET_NO_ANISOTROPY
#endif
};
#endif


//
// Functions
//

//
// This seems to work almost as well as the full-on "complicated fresnel"
// A good rindexRatio for car paint is 0.6667 (which assumes air = 1.0, clearcoat = 1.5)
//
float CalculateFresnel( float3 _eye, float3 _normal, float rindexRatio )
{
	// Note: compute R0 on the CPU and provide as a
	// constant; it is more efficient than computing R0 in
	// the vertex shader. R0 is:
	// float const R0 = pow(1.0-refractionIndexRatio, 2.0)
	// / pow(1.0+refractionIndexRatio, 2.0);
	// light and normal are assumed to be normalized
	float R0 = pow( 1.0f - rindexRatio, 2.0f) / pow ( 1.0f + rindexRatio, 2.0f );

	return R0 + ( 1.0f - R0 ) * pow( 1.0f - saturate( dot( -_eye, _normal ) ), 5.0f );
}


//-----------------------------------------------------------------------
//
// Vertex Shader(s)
//

// Input structure
struct VSINPUT
{
	float3 position : POSITION;									// Object space position
#ifdef _3DSMAX_
	float3 colour   : TEXCOORD1;								// Vertex colour
	float2 texCoord : TEXCOORD0;							// UV channel 1 texture coord - N.B. MAx requires that texcoord0 is a geometric channel
																												// as it implicitly uses that to calculate the tangent space coordinate frame.
	float3 alpha		: TEXCOORD2;								// Vertex alpha
#else
	float4 colour   : COLOR0;											// Vertex colour
	float2 texCoord : TEXCOORD0;							// UV channel 1 texture coord
#endif
};


// Output structure
struct VSOUTPUT
{
	float4 position		: POSITION;							// View-coords position
	float4 colour			: TEXCOORD1;						// Ambient lighting colour
	float2 texCoord	: TEXCOORD0;						// UV coords for texture channel 0
	float4 eye				  : TEXCOORD2;						// Eye vector (world space)
	float4 viewPos		: TEXCOORD3;						// View-coords position (copy for the PS)
	float4 vertexColour : TEXCOORD4;			// Vertex colour

	DECLARE_LIGHTING_INTERPOLATORS_VS( 5 )
};


//-----------------------------------------------------------------------
//
// Vertex shader code
//

VSOUTPUT GarageFloorVertexShader( VSINPUT _input )
{
	VSOUTPUT _output;

#if !defined( _3DSMAX_ ) && !defined(_PS3_)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	// Copy simple invariant params to output structure
#if defined( _3DSMAX_ )
	_output.colour = float4( 1.0f, 1.0f, 1.0f, 1.0f );
	_output.vertexColour.rgb = _input.colour;
	_output.vertexColour.a = _input.alpha.r;
#else
	_output.colour = float4( 1.0f, 1.0f, 1.0f, 1.0f );
	_output.vertexColour = _input.colour;
#endif
	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );
	_output.viewPos = _output.position;

	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

	// Calculate world-space vector to the eye
#ifdef _3DSMAX_
	float3 worldEyeVec = viewI[ 3 ] - worldPos;
#else
	float3 worldEyeVec = worldCameraPos - worldPos;
#endif
	_output.eye = float4(worldEyeVec,0);

	float3 normal = float3( 0.0f, 1.0f, 0.0f );

	float globalSpecularFactorValue = globalSpecularFactor;

	// Do lighting
	DO_VS_LIGHTING_CALCULATIONS

	return _output;
}



//-----------------------------------------------------------------------
//
// Fragment Shader(s)
//

// Input structure

#if defined( _3DSMAX_ )
// Max can't handle centroid interpolators properly

struct PSINPUT
{
	float4 colour			: TEXCOORD1;						// Vertex colour
	float2 texCoord	: TEXCOORD0;						// UV coords for texture channel 0
	float4 eye				  : TEXCOORD2;						// Eye vector (world space)
	float4 viewPos		: TEXCOORD3;						// View-coords position (copy for the PS)
	float4 vertexColour : TEXCOORD4;			// Vertex colour

	DECLARE_LIGHTING_INTERPOLATORS_VS( 5 )
};

#else

// Input structure
struct PSINPUT
{
	float4 colour			: TEXCOORD1;									// Vertex colour
	float2 texCoord	: TEXCOORD0;									// UV coords for texture channel 0
	float4 eye					: TEXCOORD2_centroid;		// Eye vector (world space)
	float4 viewPos		: TEXCOORD3;						// View-coords position (copy for the PS)
	float4 vertexColour : TEXCOORD4;			// Vertex colour

	DECLARE_LIGHTING_INTERPOLATORS_VS( 5 )
	DECLARE_SHADOW_PS_INPUTS
};

#endif	// defined( _3DSMAX_ )


// Output structure
struct PSOUTPUT
{
	COLOUR_OUTPUT_TYPE Colour : COLOR0;
};



//-----------------------------------------------------------------------
//
// Fragment shader code
//

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT GarageFloorFragmentShader( PSINPUT _input )
{
	PSOUTPUT _output;

	PS_GENERATE_WORLDPOS( _input.eye.xyz )

	// Read textures
	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );
	float4 specularTexColour = tex2D( specularMap, _input.texCoord );

	float globalSpecularFactorValue = globalSpecularFactor;
	float minSpecPowerValue = minSpecPower;
	float maxSpecPowerValue = maxSpecPower;

  // Normalise interpolated vectors
	float3 TSnormal = float3( 0.0f, 1.0f, 0.0f );
  float3 eye = normalize( _input.eye.xyz );
	float3 normal;

	// Do tangent space normal mapping if required
	DEPENDENT_CODE_START( useNormalMap )
#if defined( _3DSMAX_ ) || defined( NORMAL_MAPPING )
		// Because this shader is used for the special case of a flat floor mesh in a specific scene, we'll use a fixed
		// tangent space
		float3 tangent = float3( 1.0f, 0.0f, 0.0f );
		float3 binormal = float3( 0.0f, 0.0f, 1.0f );

		// Fetch and decode the map normal
		float4 normalMapColour = tex2D( normalMap, _input.texCoord );
		float3 normalFromMap = normalize( ( normalMapColour.rgb * 2.0f ) - 1.0f );

		// Perturb the tangent space normal by the normal map
		normal = ( TSnormal * normalFromMap.z ) + ( normalFromMap.x * binormal ) + ( normalFromMap.y * tangent );
		normal = normalize( normal );
#endif
	DEPENDENT_CODE_ELSE( useNormalMap )
#if defined( _3DSMAX_ ) || !defined( NORMAL_MAPPING )
		// No normal map, so use interpolated normal
		normal = TSnormal;
#endif
	DEPENDENT_CODE_END( useNormalMap )

#if !defined( _3DSMAX_ )
	// Calculate ambient shadow coord
	float2 ambientShadowCoord = ( worldPos.xz / float2( ambientXSize, ambientZSize ) ) + float2( 0.5f, 0.5f );

	// Sample ambient shadow texture
	float ambientShadowStrength = ( 1.0f - ( tex2D( ambientShadowMap, ambientShadowCoord ).g * AMBIENT_SHADOW_STRENGTH ) );

	// Modulate the diffuse colour by the ambient shadow strength (this means the ambient shadow will affect diffuse lighting which
	// isn't correct, but is what people seem to like)
	diffuseTexColour.rgb *= ambientShadowStrength;
	specularTexColour.rgb *= ambientShadowStrength;
#endif	// !defined( _3DSMAX_ )

	//
	// Calculate the fresnel factor
	//
	float fresnelCoefficient = CalculateFresnel( -eye, normal, fresnelFactor );

	// Modify the diffuse colour by the fresnel factor too
	float diffuseFresnelCoefficient = CalculateFresnel( -eye, normal, 0.8f );
	diffuseTexColour.xyz *= diffuseFresnelCoefficient;

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

#if !defined( _3DSMAX_ )
	//
	// Calculate the reflection
	//

	// Fetch the environment map colour using the world coords vector
	float2 reflectionTexCoord = ( _input.viewPos.xy / ( _input.viewPos.w * 2.0f ) ) + 0.5f;

	float4 reflectionColour = tex2D( reflectionMap, reflectionTexCoord );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (-2.0f / 512.0f), (2.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (-1.0f / 512.0f), (2.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (0.0f / 512.0f), (2.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (1.0f / 512.0f), (2.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (2.0f / 512.0f), (2.0f / 512.0f) ) );
	
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (-2.0f / 512.0f), (1.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (-1.0f / 512.0f), (1.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (0.0f / 512.0f), (1.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (1.0f / 512.0f), (1.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (2.0f / 512.0f), (1.0f / 512.0f) ) );

	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (-2.0f / 512.0f), (0.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (-1.0f / 512.0f), (0.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (1.0f / 512.0f), (0.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (2.0f / 512.0f), (0.0f / 512.0f) ) );

	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (-2.0f / 512.0f), (-1.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (-1.0f / 512.0f), (-1.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (0.0f / 512.0f), (-1.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (1.0f / 512.0f), (-1.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (2.0f / 512.0f), (-1.0f / 512.0f) ) );

	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (-2.0f / 512.0f), (-2.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (-1.0f / 512.0f), (-2.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (0.0f / 512.0f), (-2.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (1.0f / 512.0f), (-2.0f / 512.0f) ) );
	reflectionColour += tex2D( reflectionMap, reflectionTexCoord + float2( (2.0f / 512.0f), (-2.0f / 512.0f) ) );
	reflectionColour /= 25.0f;

	// Calculate envmap colour and add to diffuse
	accumulatedColour += specularTexColour * reflectionColour * reflectionFactor * fresnelCoefficient;
#endif

	// Perform lighting
	DO_PS_LIGHTING_CALCULATIONS( accumulatedColour , _input.eye.xyz )

	// Apply vertex colour to everything
	accumulatedColour *= _input.vertexColour;

	accumulatedColour.w = 1.0f;
	_output.Colour = CalculateOutputPixel( accumulatedColour );

	return _output;
}


//
// Low Detail Shaders
//


struct VSOUTPUT_LD
{
	float4 position		: POSITION;													// View-coords position
	float4 colour			: TEXCOORD1;												// Vertex colour
	float2 texCoord	: TEXCOORD0;												// UV coords for texture channel 0
};


VSOUTPUT_LD GarageFloorLowDetailVertexShader( VSINPUT _input )
{
	VSOUTPUT_LD _output;

#if !defined( _3DSMAX_ ) && !defined(_PS3_)
	float4x4	worldviewproj = mul( world, viewproj );
#endif

	_output.texCoord = _input.texCoord;

	// Calculate clip-space position of the vertex
	_output.position = mul( float4( _input.position, 1.0f ), worldviewproj );

	// get normal in world space and do lighting
	float3 normal = float3( 0.0f, 1.0f, 0.0f );
	
	// Calculate vert's world position
	float3 worldPos = mul( float4( _input.position, 1.0f ), world ).xyz;

#if defined( _3DSMAX_ )
	_output.colour.rgb = _input.colour;
	_output.colour.a = _input.alpha.r;
#else
	_output.colour = _input.colour;
#endif
	DO_VERTEX_LIGHTING( worldPos, normal, _output.colour )

	// keep alpha value from the vertex
#if defined( _3DSMAX_ )
	_output.colour.a = _input.alpha.r;
#else
	_output.colour.a = _input.colour.a;
#endif

	return _output;
}

REMOVE_UNUSED_INTERPOLATORS
PSOUTPUT GarageFloorLowDetailFragmentShader( VSOUTPUT_LD _input )
{
	PSOUTPUT _output;

	float4 diffuseTexColour = tex2D( diffuseMap, _input.texCoord );

	// Factor vertex alpha into the diffuse alpha
	diffuseTexColour.a *= _input.colour.a;

	// Calculate base colour
	float4 accumulatedColour = diffuseTexColour * _input.colour;

	accumulatedColour.w = diffuseTexColour.w;
	_output.Colour = CalculateLowDetailOutputPixel( accumulatedColour );

	return _output;
}



//-----------------------------------------------------------------------
//
// Technique(s)
//

technique GarageFloor
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "GarageFloor";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
	string lowDetailBehaviour	= "ERMB_RENDER";
	string lowDetailTechnique	= "GarageFloor_LowDetail";
	int    lowDetailDeferredID = 0;
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx GarageFloorVertexShader();
		PixelShader = compile sce_fp_rsx GarageFloorFragmentShader();
#else
		VertexShader = compile vs_3_0 GarageFloorVertexShader();
		PixelShader = compile ps_3_0 GarageFloorFragmentShader();
#endif
	}
}


technique GarageFloor_LowDetail
<
	bool supportsSpecialisedLighting = true;
	bool preservesGlobalState = true;
	string normalBehaviour		= "ERMB_RENDER";
	string normalTechnique		= "GarageFloor_LowDetail";
	int    normalDeferredID		= 0;
	string zprimeBehaviour		= "ERMB_RENDER_DEFAULT";
	string zprimeDOFBehaviour	= "ERMB_RENDER_DEFAULT";
	string shadowGenBehaviour = "ERMB_RENDER_DEFAULT";
>
{
	pass Pass0
#ifdef _3DSMAX_
	<
		bool ZEnable = true;
		bool ZWriteEnable = true;
		bool AlphaBlendEnable = false;
	>
#endif
	{
#ifdef _3DSMAX_
		ZEnable = true;
		ZWriteEnable = true;
		AlphaBlendEnable = false;
#endif

#if defined (_PS3_)
		VertexShader = compile sce_vp_rsx GarageFloorLowDetailVertexShader();
		PixelShader = compile sce_fp_rsx GarageFloorLowDetailFragmentShader();
#else
		VertexShader = compile vs_3_0 GarageFloorLowDetailVertexShader();
		PixelShader = compile ps_3_0 GarageFloorLowDetailFragmentShader();
#endif
	}
}
